"use client"

import { useEffect, useState } from "react"
import Link from "next/link"
import { Button } from "@/components/ui/button"
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card"
import { Badge } from "@/components/ui/badge"
import { ArrowLeft, Check, Clock } from "lucide-react"
import { useRouter } from "next/navigation"
import { ClientAdminLayout } from "@/components/admin/client-layout"
import dataService, { type Registration } from "@/app/lib/data-service"
import { StatusUpdateForm } from "@/components/admin/status-update-form"

export default function RegistrationDetailPage({
  params,
}: {
  params: { id: string }
}) {
  const router = useRouter()
  const [registration, setRegistration] = useState<Registration | null>(null)
  const [isLoading, setIsLoading] = useState(true)

  useEffect(() => {
    // Load registration
    const data = dataService.getRegistrationById(params.id)

    if (!data) {
      // Registration not found, redirect to registrations list
      router.push("/admin/registrations")
      return
    }

    setRegistration(data)
    setIsLoading(false)
  }, [params.id, router])

  const handleStatusUpdate = async (id: string, status: string) => {
    if (!registration) return { success: false, message: "Rekisteröintiä ei löydy" }

    try {
      const updated = dataService.updateRegistration(id, { status })

      if (updated) {
        setRegistration(updated)
        return { success: true, message: "Tila päivitetty onnistuneesti" }
      } else {
        return { success: false, message: "Rekisteröintiä ei löydy" }
      }
    } catch (error) {
      console.error("Error updating registration status:", error)
      return { success: false, message: "Tilan päivitys epäonnistui" }
    }
  }

  if (isLoading) {
    return (
      <ClientAdminLayout>
        <div className="p-6">Loading...</div>
      </ClientAdminLayout>
    )
  }

  if (!registration) {
    return (
      <ClientAdminLayout>
        <div className="p-6">Rekisteröintiä ei löydy</div>
      </ClientAdminLayout>
    )
  }

  return (
    <ClientAdminLayout>
      <div className="p-6">
        <div className="flex items-center gap-4 mb-6">
          <Link href="/admin/registrations">
            <Button variant="outline" size="sm">
              <ArrowLeft className="h-4 w-4 mr-1" />
              Takaisin
            </Button>
          </Link>
          <h1 className="text-2xl font-bold">Rekisteröinnin tiedot</h1>
          <Badge
            variant={registration.status === "pending" ? "outline" : "default"}
            className={registration.status === "pending" ? "bg-yellow-100 text-yellow-800 hover:bg-yellow-100" : ""}
          >
            {registration.status === "pending" ? "Odottaa" : "Käsitelty"}
          </Badge>
        </div>

        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
          <div className="lg:col-span-2 space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Lapsen tiedot</CardTitle>
              </CardHeader>
              <CardContent>
                <dl className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Nimi</dt>
                    <dd className="mt-1">{registration.childName}</dd>
                  </div>
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Syntymäaika</dt>
                    <dd className="mt-1">{new Date(registration.childBirthdate).toLocaleDateString("fi-FI")}</dd>
                  </div>
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Aloituspäivä</dt>
                    <dd className="mt-1">{new Date(registration.startDate).toLocaleDateString("fi-FI")}</dd>
                  </div>
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Hoitomuoto</dt>
                    <dd className="mt-1">{registration.careType === "full" ? "Kokopäivähoito" : "Osapäivähoito"}</dd>
                  </div>
                </dl>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Huoltajan tiedot</CardTitle>
              </CardHeader>
              <CardContent>
                <dl className="grid grid-cols-1 md:grid-cols-2 gap-4">
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Nimi</dt>
                    <dd className="mt-1">{registration.guardianName}</dd>
                  </div>
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Sähköposti</dt>
                    <dd className="mt-1">{registration.email}</dd>
                  </div>
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Puhelinnumero</dt>
                    <dd className="mt-1">{registration.phone}</dd>
                  </div>
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Osoite</dt>
                    <dd className="mt-1">{registration.address}</dd>
                  </div>
                </dl>
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Lisätiedot</CardTitle>
              </CardHeader>
              <CardContent>
                <dl className="space-y-4">
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Erityisruokavalio</dt>
                    <dd className="mt-1">{registration.specialDiet || "Ei erityisruokavaliota"}</dd>
                  </div>
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Allergiat</dt>
                    <dd className="mt-1">{registration.allergies || "Ei allergioita"}</dd>
                  </div>
                  <div>
                    <dt className="text-sm font-medium text-gray-500">Muut lisätiedot</dt>
                    <dd className="mt-1">{registration.additionalInfo || "Ei lisätietoja"}</dd>
                  </div>
                </dl>
              </CardContent>
            </Card>
          </div>

          <div className="space-y-6">
            <Card>
              <CardHeader>
                <CardTitle>Tila</CardTitle>
              </CardHeader>
              <CardContent>
                <StatusUpdateForm
                  registrationId={registration.id}
                  currentStatus={registration.status}
                  updateStatus={handleStatusUpdate}
                />
              </CardContent>
            </Card>

            <Card>
              <CardHeader>
                <CardTitle>Aikajana</CardTitle>
              </CardHeader>
              <CardContent>
                <div className="space-y-4">
                  <div className="flex items-start gap-3">
                    <div className="mt-1">
                      <Clock className="h-5 w-5 text-blue-500" />
                    </div>
                    <div>
                      <p className="font-medium">Rekisteröinti vastaanotettu</p>
                      <p className="text-sm text-muted-foreground">
                        {registration.createdAt
                          ? new Date(registration.createdAt).toLocaleDateString("fi-FI", {
                              day: "numeric",
                              month: "numeric",
                              year: "numeric",
                              hour: "2-digit",
                              minute: "2-digit",
                            })
                          : "Ei tietoa"}
                      </p>
                    </div>
                  </div>

                  {registration.status !== "pending" && (
                    <div className="flex items-start gap-3">
                      <div className="mt-1">
                        <Check className="h-5 w-5 text-green-500" />
                      </div>
                      <div>
                        <p className="font-medium">Rekisteröinti käsitelty</p>
                        <p className="text-sm text-muted-foreground">
                          {/* In a real app, you would store the timestamp when status was updated */}
                          Ei tarkkaa aikaleimaa
                        </p>
                      </div>
                    </div>
                  )}
                </div>
              </CardContent>
            </Card>
          </div>
        </div>
      </div>
    </ClientAdminLayout>
  )
}
